package builder;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import tree.*;
import vartab.*;

/**
 * Builds the internal data structures. These are the abstract syntax tree
 * representing the code and the symbol table representing the /home/erich/Lehre/paradigmen/workspacedeclarations.
 * This information is stored in a <code>Program</code> object.
 */
public class ASTBuilder {
    private final Program thisProgram = new Program();
    private final INode NULLStatement = new NullStatement();
    private Frame thisContext = thisProgram.getGlobalFrame();
    private Functions functions = thisProgram.getFunctions();
    private FunctionEntry thisFunction;

    public boolean isNoNullStatement(INode n) {
        return n != NULLStatement;
    }
    
    public Program buildProgram(List<INode> l) {
        thisProgram.setBlock(l);
        return thisProgram;
    }

    public INode declareVariables(Type type, List<Pair<INode>> decls) {
        BlockContext s = thisContext.topScope();
        List<INode> stmts = new ArrayList<INode>();
        for (Pair<INode> d : decls) {
            s.put(new VariableEntry(d.name, type));
            if (d.value != NULLStatement)
                stmts.add(new AssignStmt(new VarExpr(new VarRef(s, d.name)),
                        d.value));
        }
        return (stmts.size() != 0) ? new StatementSequence(stmts) : NULLStatement;
    }

    public Type declareArray(Type elementType) {
        return new ArrayType(elementType);
    }

    public INode buildNewOp(Type t, INode expr, int extraDims) {
        for (int i = 0; i < extraDims; i++)
            t = new ArrayType(t);
        return new NewOp(t, expr);
    }

    public void buildFunction(String name, Type returnType,
            List<Pair<Type>> arguments) {
        thisContext = Frame.createLocal(thisProgram.topScope());

        Type[] argumentTypes = new Type[arguments.size()];
        int i = 0;
        for (Pair<Type> a : arguments) {
            argumentTypes[i++] = a.value;
            thisContext.topScope().put(new VariableEntry(a.name, a.value));
        }
        thisFunction = new FunctionEntry(name, returnType, argumentTypes);
        functions.put(thisFunction);

    }
    
    private void removeNullStatements(List<INode> statements) {
        Iterator<INode> iter = statements.iterator();
        while (iter.hasNext()) {
            if (iter.next() == NULLStatement) iter.remove(); 
        }
    }

    public INode closeFunction(List<INode> statements) {
        removeNullStatements(statements);
        thisFunction.setBlock(new StatementSequence(statements));
        thisFunction.setVarCount(thisContext.numberOfVariables());
        thisContext = thisProgram.getGlobalFrame();
        return NULLStatement;
    }

    public void buildBlock() {
        thisContext.newContext();
    }
    
    public INode buildNullStatement() {
        return NULLStatement;
    }
    
    public INode closeBlock(List<INode> statements) {
        thisContext.leaveScope();
        removeNullStatements(statements);
        return new StatementSequence(statements);
    }

    public INode buildStringLit(String stringLiteral) {
        thisProgram.getLiterals().put(stringLiteral);
        return new StringLiteral(stringLiteral);
    }

    public INode buildWhile(INode condition, INode statement) {
        return new WhileStmt(condition, statement);
    }

    public INode buildDoWhile(INode condition, INode statement) {
        return new DoWhileStmt(condition, statement);
    }

    public INode buildIf(INode condition, INode thenPart) {
        return new IfStmt(condition, thenPart, null);
    }

    public INode buildIfElse(INode condition, INode thenPart, INode elsePart) {
        return new IfStmt(condition, thenPart, elsePart);
    }

    public INode buildCall(String n, List<INode> a) {
        return new FunctionCall(n, a);
    }

    public INode buildIndex(INode v, INode e) {
        return new IndexExpr(v, e);
    }

    public INode buildVarRef(String id) {
        return new VarRef(thisContext.topScope(), id);
    }

    public INode buildVarExpr(INode v) {
        return new VarExpr(v);
    }

    public INode buildRead(Type t, INode p) {
        return new ReadNode(t, p);
    }

    public INode buildNEQ(INode e1, INode e2) {
        return new BinOp(Codes.NEQ_OP, e1, e2);
    }

    public INode buildLTEQ(INode e1, INode e2) {
        return new BinOp(Codes.LTEQ_OP, e1, e2);
    }

    public INode buildLT(INode e1, INode e2) {
        return new BinOp(Codes.LT_OP, e1, e2);
    }

    public INode buildEQEQ(INode e1, INode e2) {
        return new BinOp(Codes.EQEQ_OP, e1, e2);
    }

    public INode buildGT(INode e1, INode e2) {
        return new BinOp(Codes.GT_OP, e1, e2);
    }

    public INode buildGTEQ(INode e1, INode e2) {
        return new BinOp(Codes.GTEQ_OP, e1, e2);
    }
    
    private static Map<String, Integer> operation =
        new HashMap<String, Integer>();
    static {
        operation.put("+", Codes.PLUS_OP);
        operation.put("-", Codes.MINUS_OP);
        operation.put("*", Codes.MULT_OP);
        operation.put("/", Codes.DIV_OP);
        operation.put("%", Codes.MOD_OP);
    }
    public INode buildOpEq(String op, INode var, INode expr) {
        INode rhs = var ;
        int oper = operation.get(op);
        return new AssignStmt((VarExpr) var, new BinOp(oper, rhs, expr));
    }

    public INode buildMinus(INode e1, INode e2) {
        return new BinOp(Codes.MINUS_OP, e1, e2);
    }

    public INode buildPlus(INode e1, INode e2) {
        return new BinOp(Codes.PLUS_OP, e1, e2);
    }

    public INode buildMod(INode e1, INode e2) {
        return new BinOp(Codes.MOD_OP, e1, e2);
    }

    public INode buildDiv(INode e1, INode e2) {
        return new BinOp(Codes.DIV_OP, e1, e2);
    }

    public INode buildMult(INode e1, INode e2) {
        return new BinOp(Codes.MULT_OP, e1, e2);
    }

    public INode buildNot(INode e) {
        return new UnOp(Codes.NOT_OP, e);
    }

    public INode buildAnd(INode e1, INode e2) {
        return new BinOp(Codes.AND_OP, e1, e2);
    }

    public INode buildOr(INode e1, INode e2) {
        return new BinOp(Codes.OR_OP, e1, e2);
    }

    public INode buildSign(INode e) {
        return new UnOp(Codes.CHS_OP, e);
    }

    public INode buildLenOp(INode v) {
        return new UnOp(Codes.LEN_OP, v);
    }

    public INode buildIntLit(Integer n) {
        return new IntLiteral(n);
    }

    public INode buildReturn(INode result) {
        return new ReturnStmt(result);
    }

    public INode buildPrintln(INode f, List<INode> v) {
        return new PrintStmt(f, v, true);
    }

    public INode buildPrint(INode f, List<INode> v) {
        return new PrintStmt(f, v, false);
    }

    public INode buildAssign(INode v, INode e) {
        return new AssignStmt((VarExpr) v, e);
    }

    public INode buildCallStmt(INode s) {
        return new CallStmt(s);
    }

    public INode dumpFunctions() {
        Dump.dump(System.err, functions);
        return NULLStatement;
    }

    public INode dumpVariables(String prompt) {
        Dump.dump(System.err, prompt, thisContext.topScope());
        return NULLStatement;
    }
}
