package tree;

import java.io.PrintStream;

public final class PrintVisitor implements IVisitor {
    private int indentationWidth = 0;
    private static final int STEP = 2;
    private final PrintStream out;

    public PrintVisitor(PrintStream out) {
        this.out = out;
    }

    /*
     * Helper functions
     */

    /**
     * Prints a number of <code>indentationWidth</code> spaces.
     */
    private void indentation() {
        for (int i = 0; i < indentationWidth; ++i)
            out.print(" ");
    }

    /**
     * Increases indentation level.
     */
    private void indent() {
        indentationWidth += STEP;
    }

    /**
     * Decreases indentation level.
     *
     */
    private void unindent() {
        indentationWidth -= STEP;
    }

    /**
     * Prints an indented string.
     * 
     * @param s String to be written.
     */
    private void put(String s) {
        indentation();
        out.print(s);
    }

    /**
     * Prints an indented string including line feed.
     * 
     * @param s String to be written.
     */
    private void putln(String s) {
        indentation();
        out.println(s);
    }

    public void visit(Program b) {
        b.stmts.accept(this);
    }

    public void visit(StatementSequence b) {
        putln("{");
        indent();
        for (INode n : b.stmts)
            n.accept(this);
        unindent();
        putln("}");
    }

    public void visit(IfStmt s) {
        put("if (");
        s.condition.accept(this);
        out.println(")");
        indent();
        s.thenPart.accept(this);
        unindent();
        if (s.elsePart != null) {
            putln("else {");
            indent();
            s.elsePart.accept(this);
            unindent();
        }
    }

    public void visit(WhileStmt s) {
        put("while (");
        s.condition.accept(this);
        out.println(")");
        indent();
        s.body.accept(this);
        unindent();
    }

    public void visit(DoWhileStmt s) {
        putln("do {");
        indent();
        s.body.accept(this);
        unindent();
        put("} while (");
        s.condition.accept(this);
        out.println(")");
    }

    public void visit(ReturnStmt s) {
        put("return ");
        if (s.expression != null)
            s.expression.accept(this);
        out.println(";");
    }

    public void visit(AssignStmt s) {
        put("");
        s.leftHandSide.accept(this);
        out.print(" = ");
        s.rightHandSide.accept(this);
        out.println(";");
    }

    public void visit(IndexExpr n) {
        n.reference.accept(this);
        out.print("[");
        n.index.accept(this);
        out.print("]");
    }

    public void visit(PrintStmt s) {
        if (s.printNewLine)
            put("println");
        else
            put("print");
        if (s.format != null) {
            out.print("[");
            s.format.accept(this);
            out.print("]");
        }
        out.print(" ");
        boolean first = true;
        for (INode n : s.expressions) {
            if (first)
                first = false;
            else
                out.print(", ");
            n.accept(this);
        }
        out.println(";");
    }

    public void visit(ReadNode s) {
        out.print("read ");
        if (s.type == Codes.INT_TYPE)
            out.print("int ");
        else
            out.print("string ");
        if (s.prompt != null) {
            out.print("(");
            s.prompt.accept(this);
            out.print(");");
        }
    }

    public void visit(NewOp n) {
        out.print("new ");
        out.print(n.type);
        out.print("[");
        n.size.accept(this);
        out.print("]");
    }

    public void visit(VarExpr e) {
        e.reference.accept(this);
    }

    public void visit(VarRef v) {
        out.print(v.name());
    }

    public void visit(CallStmt f) {
        put("");
        f.functionCall.accept(this);
        out.println(";");
    }

    public void visit(FunctionCall f) {
        out.print(f.name);
        out.print("(");
        String sep = "";
        for (INode n : f.arguments) {
            out.print(sep);
            sep = ", ";
            n.accept(this);
        }
        out.print(")");
    }

    public void visit(BinOp op) {
        op.left.accept(this);
        out.print(Codes.mem[op.operator]);
        op.right.accept(this);
    }

    public void visit(UnOp op) {
        out.print(Codes.mem[op.operator]);
        op.operand.accept(this);
    }

    public void visit(IntLiteral lit) {
        out.print(lit.number);
    }

    public void visit(StringLiteral lit) {
        out.print("\"");
        out.print(lit.string);
        out.print("\"");
    }
}