package tree;

import vartab.FunctionEntry;

/**
 * This abstract class provides an implementation of the visitor interface. It
 * traverses the AST without performing any operation. It is intended to be
 * sublassed when only a small number of visitor methods has to perform some
 * specific task. The appropriate methods are then to be overriden.
 */
public abstract class VisitorAdaptor implements IVisitor {

    public void visit(Program p) {
        p.stmts.accept(this);
        for (FunctionEntry f : p.listOfFunctions)
            f.getBlock().accept(this);
    }

    public void visit(StatementSequence b) {
        for (INode n : b.stmts)
            n.accept(this);
    }

    public void visit(IfStmt s) {
        s.condition.accept(this);
        s.thenPart.accept(this);
        if (s.elsePart != null)
            s.elsePart.accept(this);
    }

    public void visit(WhileStmt s) {
        s.condition.accept(this);
        s.body.accept(this);
    }

    public void visit(DoWhileStmt s) {
        s.condition.accept(this);
        s.body.accept(this);
    }

    public void visit(ReturnStmt s) {
        if (s.expression != null)
            s.expression.accept(this);
    }

    public void visit(AssignStmt s) {
        s.leftHandSide.accept(this);
        s.rightHandSide.accept(this);
    }

    public void visit(IndexExpr n) {
        n.reference.accept(this);
        n.index.accept(this);
    }

    public void visit(PrintStmt s) {
        if (s.format != null)
            s.format.accept(this);
        for (INode n : s.expressions)
            n.accept(this);
    }

    public void visit(ReadNode s) {
        if (s.prompt != null)
            s.prompt.accept(this);
    }

    public void visit(NewOp n) {
        n.size.accept(this);
    }

    public void visit(VarRef v) {
    }

    public void visit(VarExpr e) {
        e.reference.accept(this);
    }

    public void visit(CallStmt f) {
        f.functionCall.accept(this);
    }

    public void visit(FunctionCall f) {
        for (INode n : f.arguments)
            n.accept(this);
    }

    public void visit(BinOp op) {
        op.left.accept(this);
        op.right.accept(this);
    }

    public void visit(UnOp op) {
        op.operand.accept(this);
    }

    public void visit(IntLiteral lit) {
    }

    public void visit(StringLiteral lit) {
    }
}