package builder;

import java.util.Iterator;
import java.util.List;

import tree.AssignStmt;
import tree.BinOp;
import tree.CallStmt;
import tree.Codes;
import tree.FunctionCall;
import tree.INode;
import tree.IfStmt;
import tree.IntLiteral;
import tree.NullStatement;
import tree.PrintStmt;
import tree.Program;
import tree.ReadNode;
import tree.ReturnStmt;
import tree.StatementSequence;
import tree.StringLiteral;
import tree.UnOp;
import tree.VarExpr;
import tree.VarRef;
import tree.WhileStmt;
import vartab.BlockContext;
import vartab.Frame;
import vartab.FunctionEntry;
import vartab.Functions;
import vartab.Type;
import vartab.VariableEntry;

/**
 * Builds the internal data structures. These are the abstract syntax tree
 * representing the code and the symbol table representing the /home/erich/Lehre/paradigmen/workspacedeclarations.
 * This information is stored in a <code>Program</code> object.
 */
public class ASTBuilder {
    private final Program thisProgram = new Program();
    private final INode NULLStatement = new NullStatement();
    private Frame thisContext = thisProgram.getGlobalFrame();
    private Functions functions = thisProgram.getFunctions();
    private FunctionEntry thisFunction;

    public boolean isNoNullStatement(INode n) {
        return n != NULLStatement;
    }
    
    public Program buildProgram(List<INode> l) {
        thisProgram.setBlock(l);
        return thisProgram;
    }

    public INode declareVariable(Type type, Pair<INode> decl) {
        BlockContext s = thisContext.topScope();
        s.put(new VariableEntry(decl.name, type));
        return decl.value == NULLStatement ? 
            NULLStatement :
            new AssignStmt(new VarExpr(new VarRef(s, decl.name)), decl.value);
    }

    public void buildFunction(String name, Type returnType,
            List<Pair<Type>> arguments) {
        thisContext = Frame.createLocal(thisProgram.topScope());

        Type[] argumentTypes = new Type[arguments.size()];
        int i = 0;
        for (Pair<Type> a : arguments) {
            argumentTypes[i++] = a.value;
            thisContext.topScope().put(new VariableEntry(a.name, a.value));
        }
        thisFunction = new FunctionEntry(name, returnType, argumentTypes);
        functions.put(thisFunction);
    }
    
    private void removeNullStatements(List<INode> statements) {
        Iterator<INode> iter = statements.iterator();
        while (iter.hasNext()) {
            if (iter.next() == NULLStatement) iter.remove(); 
        }
    }

    public INode closeFunction(List<INode> statements) {
        removeNullStatements(statements);
        thisFunction.setBlock(new StatementSequence(statements));
        thisContext = thisProgram.getGlobalFrame();
        return NULLStatement;
    }

    public void buildBlock() {
        thisContext.newContext();
    }
    
    public INode buildNullStatement() {
        return NULLStatement;
    }
    
    public INode closeBlock(List<INode> statements) {
        thisContext.leaveScope();
        removeNullStatements(statements);
        return new StatementSequence(statements);
    }

    public INode buildStringLit(String stringLiteral) {
        return new StringLiteral(stringLiteral);
    }

    public INode buildWhile(INode condition, INode statement) {
        return new WhileStmt(condition, statement);
    }

    public INode buildIf(INode condition, INode thenPart) {
        return new IfStmt(condition, thenPart, null);
    }

    public INode buildIfElse(INode condition, INode thenPart, INode elsePart) {
        return new IfStmt(condition, thenPart, elsePart);
    }

    public INode buildCall(String n, List<INode> a) {
        return new FunctionCall(n, a);
    }

    public INode buildVarRef(String id) {
        return new VarRef(thisContext.topScope(), id);
    }

    public INode buildVarExpr(INode v) {
        return new VarExpr(v);
    }

    public INode buildRead(Type t, INode p) {
        return new ReadNode(t, p);
    }

    public INode buildNEQ(INode e1, INode e2) {
        return new BinOp(Codes.NEQ_OP, e1, e2);
    }

    public INode buildLTEQ(INode e1, INode e2) {
        return new BinOp(Codes.LTEQ_OP, e1, e2);
    }

    public INode buildLT(INode e1, INode e2) {
        return new BinOp(Codes.LT_OP, e1, e2);
    }

    public INode buildEQEQ(INode e1, INode e2) {
        return new BinOp(Codes.EQEQ_OP, e1, e2);
    }

    public INode buildGT(INode e1, INode e2) {
        return new BinOp(Codes.GT_OP, e1, e2);
    }

    public INode buildGTEQ(INode e1, INode e2) {
        return new BinOp(Codes.GTEQ_OP, e1, e2);
    }
    
    public INode buildMinus(INode e1, INode e2) {
        return new BinOp(Codes.MINUS_OP, e1, e2);
    }

    public INode buildPlus(INode e1, INode e2) {
        return new BinOp(Codes.PLUS_OP, e1, e2);
    }

    public INode buildMod(INode e1, INode e2) {
        return new BinOp(Codes.MOD_OP, e1, e2);
    }

    public INode buildDiv(INode e1, INode e2) {
        return new BinOp(Codes.DIV_OP, e1, e2);
    }

    public INode buildMult(INode e1, INode e2) {
        return new BinOp(Codes.MULT_OP, e1, e2);
    }

    public INode buildNot(INode e) {
        return new UnOp(Codes.NOT_OP, e);
    }

    public INode buildSign(INode e) {
        return new UnOp(Codes.CHS_OP, e);
    }

    public INode buildIntLit(Integer n) {
        return new IntLiteral(n);
    }

    public INode buildReturn(INode result) {
        return new ReturnStmt(result);
    }

    public INode buildPrintln(List<INode> v) {
        return new PrintStmt(v, true);
    }

    public INode buildPrint(List<INode> v) {
        return new PrintStmt(v, false);
    }

    public INode buildAssign(INode v, INode e) {
        return new AssignStmt((VarExpr) v, e);
    }

    public INode buildCallStmt(INode s) {
        return new CallStmt(s);
    }
    
    public INode buildNOT(INode e) {
        return new UnOp(Codes.NOT_OP, e);
    }
}
