package parser;

import gener.Parser;
import gener.Scanner;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.PrintStream;

import tree.JavaCodeGenerationVisitor;
import tree.LastCallVisitor;
import tree.Program;
import tree.ReturnCheckVisitor;
import tree.TypeCheckVisitor;
import vartab.Errors;

/**
 * Simple driver for the tinyc compiler.
 */
public class Compile {
    private static PrintStream out = System.out;

    public static void main(String argv[]) {
        if (argv.length == 0)
            System.out.println("no input file given");
        else {
            String fileName = argv[0];
            if (fileName.indexOf(".tc") != fileName.length() - 3)
                out.println(fileName + " is not a .tc  file");
            else {
                try {
                    compile(new File(fileName));
                } catch (FileNotFoundException e) {
                    out.println("File not found");
                    System.exit(1);
                } catch (IOException e) {
                    out.println("fatal IOException");
                    e.printStackTrace(out);
                } catch (Exception e) {
                    out.println("Syntax Exception");
                    e.printStackTrace(out);
                }
            }
        }
    }

    private static void compile(File inputFile) throws Exception {
        Program thisProgram = parseInput(inputFile);
        analyseProgram(thisProgram);
        String baseName = inputFile.getName();
        toJavaByteCode(baseName, thisProgram);
        out.println("No errors.");
    }

    /**
     * Syntax analysis of tinyc program.
     * 
     * @param inputFile
     *            File Object for the input file.
     * @return internal data structure of input (AST + symbol table).
     * @throws Exception
     *             when some error occured.
     */
    private static Program parseInput(File inputFile) throws Exception {
        out.println("Parsing [" + inputFile.getName() + "]");
        Scanner scanner = new Scanner(new FileReader(inputFile));
        Parser parser = new Parser(scanner);
        //Parser parser = new Parser(new DummyScanner());
        try {
            return (Program) parser.parse().value;
        } catch (Exception e) {
            out.println("Error at line: " + (scanner.lineNumber() + 1));
            throw e;
        }
    }

    /**
     * Semantic analysis of a tinyc program. This includes the type checking and
     * last call optimization.
     * 
     * @param thisProgram
     *            AST of this program.
     */
    private static void analyseProgram(Program thisProgram) {
        out.println("Type-Check");
        thisProgram.accept(new TypeCheckVisitor());
        thisProgram.accept(new ReturnCheckVisitor());
        if (Errors.getCount() != 0) {
            out.println(Errors.getCount() + " Errors");
            System.exit(1);
        }
        thisProgram.accept(new LastCallVisitor());
    }

    /**
     * Bytecode generation for java classfile format.
     * 
     * @param name
     *            base name of input file.
     * @param thisProgram
     *            AST of program.
     * @throws IOException
     *             when something goes wrong.
     */
    private static void toJavaByteCode(String name, Program thisProgram) {
        String className = name.substring(0, name.indexOf(".tc", -2));
        out.println("Generation of Java Bytecode");
        thisProgram.accept(new JavaCodeGenerationVisitor(className));
    }
}
