package fractal;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.RecursiveAction;
import javax.swing.JPanel;

public class Canvas extends JPanel implements MouseListener {
    // Anzahl paralleler Threads (> als Anzahl cores!!)
    // Vorsicht: NX muss exakt durch NTASKS teilbar sein!!
    public static final int NTASKS = 20;
    
    public static final int NX = 800;
    public static final int NY = 600;
    public static final int MC = 1000;
    private static final int COLORS = 64;
    private static final Color colors[] = new Color[COLORS];
    static {
        for (int i = 0; i < COLORS; i++)
            colors[i] = computeRGB(i);
    }
    private Color[][] area = new Color[NX][NY];
    private double x0 = -2;
    private double y0 = -1;
    private double dx = 3;
    private double dy = 2;
    private Sequence seq = new Sequence(MC);
    private ForkJoinPool pool = new ForkJoinPool();

    public Canvas() {
        super();
        if (NX % NTASKS != 0)
            throw new ArithmeticException("NTASKS muss NX teilen!");
        setPreferredSize(new Dimension(NX, NY));
        addMouseListener(this);
        newData();
        repaint();
    }

    private void newData() {
        long start = System.nanoTime();
        pool.invoke(new ForkJoinSeq(x0, 0, 0));
        long diff = System.nanoTime() - start;
        System.out.printf("Time: %.2f%n", 1e-9*diff);
    }

    public void paintComponent(Graphics g) {
        for (int i = 0; i < NX; i++) {
            for (int j = 0; j < NY; j++) {
                g.setColor(area[i][j]);
                g.drawLine(i, j, i, j);
            }
        }
    }
    
    static Color getColor(int hue) {
        return hue < MC ? colors[hue % COLORS] : new Color(0, 0, 0);
    }

    private static Color computeRGB(int hue) {
        double xhue = (double) (hue % COLORS) / COLORS * 6.0;
        double i = Math.floor(xhue);
        double f = xhue - i;
        int p = 0;
        int v = 255;
        int q = (int) (v * (1 - f));
        int t = (int) (v * f);
        switch ((int) i) {
        case 0:
            return new Color(v, t, p);
        case 1:
            return new Color(q, v, p);
        case 2:
            return new Color(p, v, t);
        case 3:
            return new Color(p, q, v);
        case 4:
            return new Color(t, p, v);
        case 5:
            return new Color(v, p, q);
        }
        return new Color(0, 0, 0);
    }

    public void mouseClicked(MouseEvent e) {
        if (e.getButton() == MouseEvent.BUTTON3) {
            x0 -= 2 * dx;
            y0 -= 2 * dy;
            dx *= 5;
            dy *= 5;
            newData();
            repaint();
        }
    }

    private int firstX;
    private int firstY;

    public void mousePressed(MouseEvent e) {
        if (e.getButton() == MouseEvent.BUTTON1) {
            firstX = e.getX();
            firstY = e.getY();
        }
    }

    public void mouseReleased(MouseEvent e) {
        if (e.getButton() == MouseEvent.BUTTON1) {
            int x = e.getX();
            int y = e.getY();
            int deltaX = Math.abs(x - firstX);
            int deltaY = Math.abs(y - firstY);
            int delta = Math.max(deltaX / 4, deltaY / 3);
            deltaX = 4 * delta;
            deltaY = 3 * delta;
            int cX = Math.min(x, firstX);
            int cY = Math.min(y, firstY);
            x0 = x0 + dx * cX / NX;
            y0 = y0 + dy * cY / NY;
            dx = dx * deltaX / NX;
            dy = dy * deltaY / NY;
            newData();
            repaint();
        }
    }

    public void mouseEntered(MouseEvent e) {}
    public void mouseExited(MouseEvent e) {}
    private static final long serialVersionUID = -1034045915033158078L;
    
    class ForkJoinSeq extends RecursiveAction { 
        private double x0;
        private int i0;
        private int nx;
        
        public ForkJoinSeq(double x0, int i0, int nx) {
            this.x0 = x0;
            this.i0 = i0;
            this.nx = nx;;
        }
        
        protected void compute() {
            if (nx == 0) {
                int n = NX / NTASKS;
                List<ForkJoinSeq> tasks = new ArrayList<ForkJoinSeq>();
                for (int i = i0; i < NX; i += n) {
                    tasks.add(new ForkJoinSeq(x0 + i * dx/NX, i, n));
                }
                invokeAll(tasks);
            } else
                computeDirectly();
        }
        
        protected void computeDirectly() {
            double x = x0;
            double deltaX = dx/NX;
            double deltaY = dy/NY;
            for (int i = i0; i < i0 + nx; i++) {
                double y = y0;
                for (int j = 0; j < NY; j++) {
                    area[i][j] = getColor(seq.next(x, y));
                    y += deltaY;
                }
                x += deltaX;
            }
        }
    }
}