package jmath;

import static java.lang.Math.*;

/**
 * Numeric integration of systems of ordinary differential equations
 * by the Runge-Kutta method.
 * <p>
 * This object contains as appliction the computation of motion
 * in a homogenous gravitational field in the presence of atmospheric drag.
 */
public final class RungeKutta {
    private RungeKutta() {}

    /**
     * One RKT-step.
     * 
     * @param x0 initial parameter
     * @param h  step-width for the parameter
     * @param y0 initial solution-vector
     * @param fv function describing the differentials
     * 
     * @return next solution vector
     */

    public static Vector step(double x0, double h, Vector y0, DVFunction2 fv) {
        Vector k1 = fv.apply(x0, y0).mult(h);
        Vector k2 = fv.apply(x0 + h / 2, y0.plus(k1.div(2.))).mult(h);
        Vector k3 = fv.apply(x0 + h / 2, y0.plus(k2.div(2.))).mult(h);
        Vector k4 = fv.apply(x0 + h, y0.plus(k3)).mult(h);
        return k1.plus(k2.plus(k3).mult(2.)).plus(k4).div(6.);
    }

    /* parameters for gravitation, body and atmosphere.
     */
    static final double g = 9.81;
    static final double cw = 0.25;
    static final double rho = 1.2;
    static final double A = 1e-4;
    static final double m = 1e-3;
    static final double r = 0.5 * cw * rho * A / m;

    static DVFunction2 drag = new DVFunction2() {
        /**
         * functional description the velocities and the accelerations.
         * 
         * @param x parameter
         * @param v initial coordinates and velocities.
         */
        public Vector apply(double x, Vector v) {
            double c = sqrt(v.at(1) * v.at(1) + v.at(3) * v.at(3));
            return new Vector(v.at(1), -r * v.at(1) * c, v.at(3), -g - r
                * v.at(3) * c);
        }
    };

    /**
     * Test driver.
     * <p>
     * The algorithmes simulates the motion with the initial conditions
     * v = 30 m/sec and alpha = 45 degrees.
     * 
     * @param args
     */
    public static void main(String[] args) {
        double alpha = Math.PI / 4.;
        double v = 30.0;
        double vx = v * cos(alpha);
        double vy = v * sin(alpha);
        int n = 210;
        double h = 4.3248 / n;

        Vector x = new Vector(0, vx, 0, vy);
        double t = 0.0;
        for (int i = 0; i <= n + 1; i++) {
            if (i % 10 == 0)
                System.out.printf("%9.3f %9.3f %9.3f%n", t, x.at(0), x.at(2));
            x = x.plus(step(t, h, x, drag));
            t += h;
        }
    }
}
