package jmath;

import java.util.Formatter;

/**
 * Class Vector represents a n-dimensional vector and its most basic
 * operations. A Vector is immutable.
 * <p>
 * Operations operating on two vectors require that both are of the same
 * dimension.
 * <p>
 * This class has been developed as demonstration of the use of higher order
 * functions.
 * 
 * @author Erich Ehses
 */
public class Vector {
    private double[] v;

    /**
     * Creates an vector defined by the arguments.
     *
     * @param xs vector components
     */    
    public Vector(double... xs) {
        v = xs;
    }

    /**
     * Access to n-th component (starting with 0).
     * 
     * @param n index
     * @return  coordinate
     */
    public double at(int n) {
        return v[n];
    }

    /**
     * Returns the dimension of the vector.
     *
     * @return dimension.
     */
    public int dim() {
        return v.length;
    }

    private Vector map2(Vector that, DFunction2 oper) {
        assert (that.dim() == this.dim());
        double[] a = new double[v.length];
        for (int i = 0; i < v.length; i++)
            a[i] = oper.apply(v[i], that.v[i]);
        return new Vector(a);
    }

    private Vector map(DFunction1 oper) {
        double[] a = new double[v.length];
        for (int i = 0; i < v.length; i++)
            a[i] = oper.apply(v[i]);
        return new Vector(a);
    }

    /**
     * The sum of all vector-components.
     *
     * @return sum
     */
    public double sum() {
        double s = 0.0;
        for (double x: v)
            s += x;
        return s;
    }

    /**
     * A vector of the sums of this and that elements.
     *
     * @param that the second operand.
     * @return sums
     */
    public Vector plus(Vector that) {
        return map2(that, new DFunction2() {
            public double apply(double x, double y) {
                return x + y;
            }
        });
    }

    /**
     * A vector of the differences of this and that elements.
     *
     * @param that the second operand.
     * @return differences
     */
    public Vector minus(Vector that) {
        return map2(that, new DFunction2() {
            public double apply(double x, double y) {
                return x - y;
            }
        });
    }

    /**
     * A vector of the products of this and that elements.
     *
     * @param that the second operand.
     * @return products
     */
    public Vector mult(Vector that) {
        return map2(that, new DFunction2() {
            public double apply(double x, double y) {
                return x * y;
            }
        });
    }

    /**
     * A vector of the quotients of this and that elements.
     *
     * @param that the second operand.
     * @return quotients
     */
    public Vector div(Vector that) {
        return map2(that, new DFunction2() {
            public double apply(double x, double y) {
                return x / y;
            }
        });
    }

    /**
     * The product of this and a scalar.
     *
     * @param scalar the second operand.
     * @return product-vector
     */
    public Vector mult(final double scalar) {
        return map(new DFunction1() {
            public double apply(double x) {
                return x * scalar;
            }
        });
    }

    /**
     * The quotient of this and a scalar.
     *
     * @param scalar the second operand.
     * @return vector divided by scalar
     */
    public Vector div(final double scalar) {
        return map(new DFunction1() {
            public double apply(double x) {
                return x / scalar;
            }
        });
    }

    /**
     * Dot-product of this and that.
     *
     * @param that
     * @return dot-product
     */
    public double dot(Vector that) {
        return this.mult(that).sum();
    }

    /**
     * Absolute value of this vector.
     *
     * @return absolute value
     */
    public double abs() {
        return Math.sqrt(map(new DFunction1() {
            public double apply(double x) {
                return x * x;
            }
        }).sum());
    }

    @Override
    public String toString() {
        Formatter f = new Formatter();
        f.format("(");
        boolean notFirst = false;
        for (double x: v) {
            if (notFirst) f.format(", ");
            f.format("%9.3f", x);
        }
        f.format(")");
        return f.toString();
    }

    @Override
    public boolean equals(Object that) {
        if (!(that instanceof Vector)) return false;
        Vector other = (Vector) that;
        return java.util.Arrays.equals(this.v, other.v);
    }
}
