package smath

import math._

/**
 * Numeric integration of systems of ordinary differential equations
 * by the Runge-Kutta method.
 * <p>
 * This object contains as appliction the computation of motion
 * in a homogenous gravitational field in the presence of atmospheric drag.
 */
object RungeKutta {
  type Fkt = (Double, Vector) => Vector

  /**
   * One RKT-step.
   * 
   * @param x0 initial parameter
   * @param h  step-width for the parameter
   * @param y0 initial solution-vector
   * @param fv function describing the differentials
   * 
   * @return next solution vector
   */
  def step(x0: Double, h: Double, y0: Vector, fv: Fkt) = {
    val k1 = fv(x0, y0) * h
    val k2 = fv(x0 + h / 2, y0 + k1 / 2.) * h
    val k3 = fv(x0 + h / 2, y0 + k2 / 2.) * h
    val k4 = fv(x0 + h, y0 + k3) * h
    (k1 + k2 * 2. + k3 * 2. + k4) / 6.
  }

  /* parameters for gravitation, body and atmosphere.
   */
  val g = 9.81
  val cw = 0.25
  val rho = 1.2
  val A = 1e-4
  val m = 1e-3
  val r = 0.5 * cw * rho * A / m

  /**
   * functional description the velocities and the accelerations.
   * 
   * @param x parameter
   * @param v initial coordinates and velocities.
   */
  def drag(x: Double, v: Vector) = {
    val c = sqrt(v(1) * v(1) + v(3) * v(3))
    Vector(v(1), -r * v(1) * c, v(3), -g - r * v(3) * c)
  }

  /**
   * Test driver.
   * <p>
   * The algorithmes simulates the motion with the initial conditions
   * v = 30 m/sec and alpha = 45 degrees.
   */
  def main(args: Array[String]) {
    val alpha = Pi / 4.
    val v = 30.0
    val vx = v * cos(alpha)
    val vy = v * sin(alpha)
    val n = 210
    val h = 4.3248 / n

    var x = Vector(0, vx, 0, vy)
    var t = 0.0
    for (i <- 0 to n + 1) {
      if (i % 10 == 0) printf("%9.3f %9.3f %9.3f%n", t, x(0), x(2))
      x += step(t, h, x, drag _)
      t += h
    }
  }
}