package smath

/**
 * Class Vector represents a n-dimensional vector and its most basic
 * operations. A Vector is immutable.
 * <p>
 * Operations operating on two vectors require that both are of the same
 * dimension.
 * <p>
 * This class has been developed as demonstration of the use of higher order
 * functions.
 * 
 * @author Erich Ehses
 */
class Vector(private val v: Array[Double]) {
  
  /**
   * Access to n-th component (starting with 0).
   * 
   * @param n index
   * @return  coordinate
   */
  def apply(n: Int) = v(n)

  /**
   * Returns the dimension of the vector.
   *
   * @return dimension.
   */
  def dim = v.length

  private def map2(that: Vector, oper: (Double, Double) => Double) = {
    require(that.dim == this.dim)
    val a = new Array[Double](v.length)
    for (i <- 0 until v.length)
      a(i) = oper(v(i), that.v(i))
    new Vector(a)
  }

  /**
   * The sum of all vector-components.
   *
   * @return sum
   */
  def sum = v.sum
  
  /**
   * A vector of the sums of this and that elements.
   *
   * @param that the second operand.
   * @return sums
   */
  def +(that: Vector) = map2(that, _ + _)
  
  /**
   * A vector of the differences of this and that elements.
   *
   * @param that the second operand.
   * @return differences
   */
  def -(that: Vector) = map2(that, _ - _)
  
  /**
   * A vector of the products of this and that elements.
   *
   * @param that the second operand.
   * @return products
   */
  def *(that: Vector) = map2(that, _ * _)
  
  /**
   * A vector of the quotients of this and that elements.
   *
   * @param that the second operand.
   * @return quotients
   */
  def /(that: Vector) = map2(that, _ / _)
  
  /**
   * The product of this and a scalar.
   *
   * @param scalar the second operand.
   * @return product-vector
   */
   def *(scalar: Double) = new Vector(v.map(_ * scalar))
  
  /**
   * The quotient of this and a scalar.
   *
   * @param scalar the second operand.
   * @return vector divided by scalar
   */
  def /(scalar: Double) = new Vector(v.map(_ / scalar))
  
  /**
   * Dot-product of this and that.
   *
   * @param that
   * @return dot-product
   */
  def dot(that: Vector) = (this * that).sum
  
  /**
   * Absolute value of this vector.
   *
   * @return absolute value
   */
  def abs = math.sqrt(v.map(x => x * x).sum)

  override def toString = v.mkString("(", ",", ")")
  override def equals(that: Any) = that match {
    case t: Vector => this.v == t.v
    case _ => false
  }
}

object Vector {
  /**
   * Factory creating an vector defined by the arguments.
   *
   * @param vs vector components
   * @return new vector
   */
  def apply(vs: Double*) = new Vector(vs.toArray)
}