package math
import scala.math.Pi

object ComputePi {
  /**
   * Eine Zahlenfolge
   */
  type Ds = Stream[Double]

  /**
   * Bestimmt unterschiedliche Approximationen fuer Pi
   */
  def main(args: Array[String]) {
    def sign(n: Int) = if (n % 2 ==1) 1 else -1
    val piSeries = partialSums(Stream.from(1).map(n => 4.0 * sign(n)/(2*n-1)))
    val shanksTransform1 = shanksTransform(piSeries)
    val shanksTransform2 = shanksTransform(shanksTransform1)
    //  ...
    val shanksTransformOpt = accSeq(piSeries, shanksTransform)
    
    val print2N = 10
    printStream("PI-Series", piSeries, print2N)
    printStream("Shanks-1", shanksTransform1, print2N)
    printStream("Shanks-2", shanksTransform2, print2N)
    printStream("optimal", shanksTransformOpt, print2N)
    printf("%nmath.Pi%n%.15f%n", Pi)
  }
  
  /**
   * Berechnet die Partialsummen einer Zahlenfolge
   * @param s Zahlenfolge
   * @return Partialsummen
   */
  def partialSums(s: Ds): Ds = {
    def partialSums(sum: Double, rest: Ds): Ds = 
      sum #:: partialSums(sum + rest.head, rest.tail)
    partialSums(s.head, s.tail)
  }

  /**
   * Berechnet die Shank's Transformierte einer Zahlenfolge
   * @param s Zahlenfolge
   * @return Transformierte
   */
  def shanksTransform(s: Ds): Ds = {
    val sd = s(2) - s(1)
    (s(2) - (sd * sd) / (s(0) - 2 * s(1) + s(2))) #:: shanksTransform(s.tail)
  }

  /**
   * Bildet fortgesetzte Transformationen einer Zahlenfolge.
   * Wenn s die Eingabefolge ist, so ist das Ergebnis die Folge s,
   * tf(s), tf(tf(s)) usw.
   * @param s Zahlenfolge
   * @param tf Transformationsfunktion
   * @return Folg der transformierten Folgen.
   */
  def makeTableau(s: Ds, tf: Ds => Ds): Stream[Ds] =
    s #:: makeTableau(tf(s), tf)
  
   /**
    * Bildet die forgesetzte Transformation einer Zahlenfolge, gibt
    * aber jeweils nur das Anfangselement jeder Folge zurueck.
    * Wenn s die Eingangsfolge ist, so ist das Ergebnis die Folge s(0),
    * tf(s)(0), tf(tf(s))(0) usw.
    * @param s Zahlenfolge
    * @param tf Transformationsfunktion
    * @return Folge der ersten Folgenelemente der Transformationen
    */
  def accSeq(s: Ds, tf: Ds => Ds): Ds = makeTableau(s, tf).map(_.head)

  /**
   * Ausgabe von n Elementen eines Streams.
   * @param title Titel der Ausgabe
   * @param s der auszugebende Stream
   * @param n die Anzahl der Elemente
   */
  def printStream(title: String, s: Ds, n: Int) {
    println("\n" + title)
    println(s.take(n).map("%.15f".format(_)).mkString("\n"))
  }
}